.rqr.fast <- function(y, x, w = NULL, G, zeta, m, initq) {
  #' rqr.fast
  #'
  #' Algorithm 2: algorithm with preprocessing for Rotated Quantile Regression 
  #' (RQR) for a grid of quantiles and the rotation obtained with a copula.
  #'
  #' @param y = Dependent variable (N x 1)
  #' @param x = Regressors matrix (N x K)
  #' @param w = Sample weights (N x 1)
  #' @param G = Copula conditional on participation (N x Q)
  #' @param zeta =  Conservative estimate of standard error of residuals (N x 1)
  #' @param m =  Parameter to select interval of observations in top and bottom
  #' groups
  #' @param initq = Initial quantile to estimate regularly and obtain 
  #' preliminary values for remaining quantiles
  #'
  #' @return b = Estimated beta coefficients (K x Q)
  
  # Number of observations (N) and regressors (K)
  N <- NROW(x)
  K <- NCOL(x)
  
  # Initialize sample weights
  if (is.null(w)) {
    w <- rep(1, N)
  }
  
  # Number of quantiles
  Q <- NCOL(G)
  
  # Pregenerate matrix to store beta parameters
  b <- matrix(0, nrow = K, ncol = Q)
  
  # Compute beta parameters for initial quantile (initial step)
  taumean <- mean(G[, initq])
  b[, initq] <- as.vector(quantreg::rq.fit.fnb(x*w,
                                     y*w,
                                     tau = taumean,
                                     rhs= t(x*w) %*% (1 - G[, initq]),
                                     beta = 0.99995, eps = 1e-06)$coefficients) 
  
  # Compute remaining beta parameters with preprocessing (steps 1-4)
  for (i1 in (initq + 1):Q) {
    b[, i1] <- .rqrtau.fast(y, x, w, G[, i1], zeta, m, b[, i1 - 1])
  }
  for (i1 in 1:(initq - 1)) {
    b[, initq - i1] <- .rqrtau.fast(y, x, w, G[, initq - i1], zeta, m, b[, initq - i1 + 1])
  }
  
  return(b)
}